<?php
/**
 * Admin Add Product
 */
define('ECOMMERCE_ACCESS', true);
require_once '../config/db.php';
require_once '../config/auth.php';

requireAdmin();

$settings = getSettings();
$errors = [];
$success = false;

// Get categories
$categories = fetchAll("SELECT * FROM categories WHERE status = 1 ORDER BY name");

// Handle form submission
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    if (!verifyCSRFToken($_POST['csrf_token'] ?? '')) {
        $errors[] = 'Invalid request. Please try again.';
    } else {
        $name = sanitize($_POST['name'] ?? '');
        $slug = sanitize($_POST['slug'] ?? '');
        $category_id = intval($_POST['category_id'] ?? 0);
        $description = sanitize($_POST['description'] ?? '');
        $short_description = sanitize($_POST['short_description'] ?? '');
        $price = floatval($_POST['price'] ?? 0);
        $sale_price = $_POST['sale_price'] ? floatval($_POST['sale_price']) : null;
        $sku = sanitize($_POST['sku'] ?? '');
        $stock = intval($_POST['stock'] ?? 0);
        $status = sanitize($_POST['status'] ?? 'draft');
        $featured = isset($_POST['featured']) ? 1 : 0;
        
        // Validation
        if (empty($name)) {
            $errors[] = 'Product name is required.';
        }
        if (empty($slug)) {
            $slug = strtolower(preg_replace('/[^a-zA-Z0-9]+/', '-', $name));
        }
        if ($price <= 0) {
            $errors[] = 'Price must be greater than 0.';
        }
        if ($sale_price !== null && $sale_price >= $price) {
            $errors[] = 'Sale price must be less than regular price.';
        }
        
        // Check slug uniqueness
        $existing = fetchOne("SELECT id FROM products WHERE slug = ?", [$slug]);
        if ($existing) {
            $slug = $slug . '-' . time();
        }
        
        // Handle image upload
        $image = null;
        if (isset($_FILES['image']) && $_FILES['image']['error'] === UPLOAD_ERR_OK) {
            $uploadDir = '../uploads/products/';
            if (!is_dir($uploadDir)) {
                mkdir($uploadDir, 0755, true);
            }
            
            $allowedTypes = ['image/jpeg', 'image/png', 'image/webp', 'image/gif'];
            $fileType = $_FILES['image']['type'];
            
            if (!in_array($fileType, $allowedTypes)) {
                $errors[] = 'Invalid image type. Allowed: JPG, PNG, WebP, GIF';
            } else {
                $ext = pathinfo($_FILES['image']['name'], PATHINFO_EXTENSION);
                $image = uniqid('product_') . '.' . $ext;
                
                if (!move_uploaded_file($_FILES['image']['tmp_name'], $uploadDir . $image)) {
                    $errors[] = 'Failed to upload image.';
                    $image = null;
                }
            }
        }
        
        // Insert product
        if (empty($errors)) {
            query(
                "INSERT INTO products (category_id, name, slug, description, short_description, price, sale_price, sku, stock, image, featured, status) 
                 VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)",
                [$category_id ?: null, $name, $slug, $description, $short_description, $price, $sale_price, $sku ?: null, $stock, $image, $featured, $status]
            );
            
            setFlash('success', 'Product added successfully!');
            header('Location: products.php');
            exit;
        }
    }
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Add Product - Admin Panel</title>
    <link rel="stylesheet" href="../assets/style.css">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
    <style>
        :root {
            --primary: #1e3a5f;
            --primary-light: #2d4a6f;
            --success: #10b981;
            --danger: #ef4444;
            --gray-100: #f3f4f6;
            --gray-200: #e5e7eb;
            --gray-500: #6b7280;
            --gray-700: #374151;
            --gray-900: #111827;
        }
        * { box-sizing: border-box; margin: 0; padding: 0; }
        body { font-family: 'Segoe UI', system-ui, sans-serif; background: var(--gray-100); color: var(--gray-700); }
        .admin-wrapper { display: flex; min-height: 100vh; }
        .sidebar { width: 260px; background: var(--primary); color: #fff; position: fixed; height: 100vh; overflow-y: auto; }
        .sidebar-header { padding: 20px; border-bottom: 1px solid rgba(255,255,255,0.1); }
        .sidebar-header h2 { font-size: 20px; font-weight: 700; }
        .sidebar-header span { font-size: 12px; opacity: 0.7; }
        .sidebar-menu { padding: 20px 0; }
        .menu-item { display: flex; align-items: center; padding: 14px 20px; color: rgba(255,255,255,0.8); text-decoration: none; transition: all 0.3s; gap: 12px; }
        .menu-item:hover, .menu-item.active { background: rgba(255,255,255,0.1); color: #fff; }
        .menu-item i { width: 20px; text-align: center; }
        .main-content { flex: 1; margin-left: 260px; padding: 30px; }
        .page-header { display: flex; justify-content: space-between; align-items: center; margin-bottom: 30px; }
        .page-header h1 { font-size: 28px; color: var(--gray-900); }
        .btn { padding: 10px 20px; border-radius: 8px; text-decoration: none; font-weight: 500; cursor: pointer; border: none; transition: all 0.3s; display: inline-flex; align-items: center; gap: 8px; }
        .btn-primary { background: var(--primary); color: #fff; }
        .btn-primary:hover { background: var(--primary-light); }
        .btn-secondary { background: var(--gray-200); color: var(--gray-700); }
        .card { background: #fff; border-radius: 12px; box-shadow: 0 2px 8px rgba(0,0,0,0.04); margin-bottom: 24px; }
        .card-header { padding: 20px 24px; border-bottom: 1px solid var(--gray-200); }
        .card-header h3 { font-size: 18px; color: var(--gray-900); }
        .card-body { padding: 24px; }
        .form-grid { display: grid; grid-template-columns: repeat(2, 1fr); gap: 24px; }
        .form-group { margin-bottom: 20px; }
        .form-group.full { grid-column: span 2; }
        .form-group label { display: block; margin-bottom: 8px; font-weight: 500; color: var(--gray-700); }
        .form-group input, .form-group select, .form-group textarea {
            width: 100%; padding: 12px 16px; border: 1px solid var(--gray-200); border-radius: 8px; font-size: 14px; transition: border-color 0.3s;
        }
        .form-group input:focus, .form-group select:focus, .form-group textarea:focus {
            outline: none; border-color: var(--primary);
        }
        .form-group textarea { min-height: 120px; resize: vertical; }
        .form-group small { color: var(--gray-500); font-size: 12px; margin-top: 4px; display: block; }
        .checkbox-group { display: flex; align-items: center; gap: 10px; }
        .checkbox-group input { width: auto; }
        .error-list { background: #fee2e2; color: #dc2626; padding: 16px 20px; border-radius: 8px; margin-bottom: 24px; }
        .error-list ul { margin: 0; padding-left: 20px; }
        .image-preview { margin-top: 10px; max-width: 200px; max-height: 200px; object-fit: cover; border-radius: 8px; display: none; }
        .form-actions { display: flex; gap: 12px; justify-content: flex-end; padding-top: 20px; border-top: 1px solid var(--gray-200); margin-top: 20px; }
        @media (max-width: 768px) {
            .sidebar { width: 100%; height: auto; position: relative; }
            .main-content { margin-left: 0; }
            .admin-wrapper { flex-direction: column; }
            .form-grid { grid-template-columns: 1fr; }
            .form-group.full { grid-column: span 1; }
        }
    </style>
</head>
<body>
    <div class="admin-wrapper">
        <!-- Sidebar -->
        <aside class="sidebar">
            <div class="sidebar-header">
                <h2><?php echo htmlspecialchars($settings['site_name']); ?></h2>
                <span>Admin Panel</span>
            </div>
            <nav class="sidebar-menu">
                <a href="dashboard.php" class="menu-item"><i class="fas fa-tachometer-alt"></i> Dashboard</a>
                <a href="products.php" class="menu-item active"><i class="fas fa-box"></i> Products</a>
                <a href="categories.php" class="menu-item"><i class="fas fa-tags"></i> Categories</a>
                <a href="orders.php" class="menu-item"><i class="fas fa-shopping-cart"></i> Orders</a>
                <a href="users.php" class="menu-item"><i class="fas fa-users"></i> Users</a>
                <a href="settings.php" class="menu-item"><i class="fas fa-cog"></i> Settings</a>
                <a href="logout.php" class="menu-item"><i class="fas fa-sign-out-alt"></i> Logout</a>
            </nav>
        </aside>

        <!-- Main Content -->
        <main class="main-content">
            <div class="page-header">
                <h1>Add Product</h1>
                <a href="products.php" class="btn btn-secondary"><i class="fas fa-arrow-left"></i> Back</a>
            </div>

            <?php if (!empty($errors)): ?>
            <div class="error-list">
                <ul>
                    <?php foreach ($errors as $error): ?>
                    <li><?php echo htmlspecialchars($error); ?></li>
                    <?php endforeach; ?>
                </ul>
            </div>
            <?php endif; ?>

            <form method="POST" enctype="multipart/form-data">
                <?php echo csrfField(); ?>
                
                <div class="card">
                    <div class="card-header">
                        <h3>Product Information</h3>
                    </div>
                    <div class="card-body">
                        <div class="form-grid">
                            <div class="form-group">
                                <label for="name">Product Name *</label>
                                <input type="text" id="name" name="name" required 
                                       value="<?php echo htmlspecialchars($_POST['name'] ?? ''); ?>">
                            </div>
                            
                            <div class="form-group">
                                <label for="slug">URL Slug</label>
                                <input type="text" id="slug" name="slug" 
                                       value="<?php echo htmlspecialchars($_POST['slug'] ?? ''); ?>">
                                <small>Leave empty to auto-generate from name</small>
                            </div>
                            
                            <div class="form-group">
                                <label for="category_id">Category</label>
                                <select id="category_id" name="category_id">
                                    <option value="">Select Category</option>
                                    <?php foreach ($categories as $cat): ?>
                                    <option value="<?php echo $cat['id']; ?>" 
                                            <?php echo (($_POST['category_id'] ?? '') == $cat['id']) ? 'selected' : ''; ?>>
                                        <?php echo htmlspecialchars($cat['name']); ?>
                                    </option>
                                    <?php endforeach; ?>
                                </select>
                            </div>
                            
                            <div class="form-group">
                                <label for="sku">SKU</label>
                                <input type="text" id="sku" name="sku" 
                                       value="<?php echo htmlspecialchars($_POST['sku'] ?? ''); ?>">
                            </div>
                            
                            <div class="form-group full">
                                <label for="short_description">Short Description</label>
                                <textarea id="short_description" name="short_description" rows="2"><?php echo htmlspecialchars($_POST['short_description'] ?? ''); ?></textarea>
                            </div>
                            
                            <div class="form-group full">
                                <label for="description">Full Description</label>
                                <textarea id="description" name="description"><?php echo htmlspecialchars($_POST['description'] ?? ''); ?></textarea>
                            </div>
                        </div>
                    </div>
                </div>

                <div class="card">
                    <div class="card-header">
                        <h3>Pricing & Inventory</h3>
                    </div>
                    <div class="card-body">
                        <div class="form-grid">
                            <div class="form-group">
                                <label for="price">Regular Price (<?php echo $settings['currency']; ?>) *</label>
                                <input type="number" id="price" name="price" step="0.01" min="0" required
                                       value="<?php echo htmlspecialchars($_POST['price'] ?? ''); ?>">
                            </div>
                            
                            <div class="form-group">
                                <label for="sale_price">Sale Price (<?php echo $settings['currency']; ?>)</label>
                                <input type="number" id="sale_price" name="sale_price" step="0.01" min="0"
                                       value="<?php echo htmlspecialchars($_POST['sale_price'] ?? ''); ?>">
                                <small>Leave empty for no sale</small>
                            </div>
                            
                            <div class="form-group">
                                <label for="stock">Stock Quantity *</label>
                                <input type="number" id="stock" name="stock" min="0" required
                                       value="<?php echo htmlspecialchars($_POST['stock'] ?? '0'); ?>">
                            </div>
                            
                            <div class="form-group">
                                <label for="status">Status</label>
                                <select id="status" name="status">
                                    <option value="draft" <?php echo (($_POST['status'] ?? 'draft') === 'draft') ? 'selected' : ''; ?>>Draft</option>
                                    <option value="active" <?php echo (($_POST['status'] ?? '') === 'active') ? 'selected' : ''; ?>>Active</option>
                                    <option value="inactive" <?php echo (($_POST['status'] ?? '') === 'inactive') ? 'selected' : ''; ?>>Inactive</option>
                                </select>
                            </div>
                        </div>
                    </div>
                </div>

                <div class="card">
                    <div class="card-header">
                        <h3>Product Image</h3>
                    </div>
                    <div class="card-body">
                        <div class="form-group">
                            <label for="image">Product Image</label>
                            <input type="file" id="image" name="image" accept="image/*" onchange="previewImage(this)">
                            <small>Recommended: 800x800px. Max 2MB. JPG, PNG, WebP, GIF</small>
                            <img id="imagePreview" class="image-preview" alt="Preview">
                        </div>
                        
                        <div class="form-group">
                            <div class="checkbox-group">
                                <input type="checkbox" id="featured" name="featured" value="1" 
                                       <?php echo isset($_POST['featured']) ? 'checked' : ''; ?>>
                                <label for="featured" style="margin-bottom: 0;">Featured Product</label>
                            </div>
                        </div>
                    </div>
                </div>

                <div class="form-actions">
                    <a href="products.php" class="btn btn-secondary">Cancel</a>
                    <button type="submit" class="btn btn-primary"><i class="fas fa-save"></i> Save Product</button>
                </div>
            </form>
        </main>
    </div>

    <script>
        // Auto-generate slug from name
        document.getElementById('name').addEventListener('input', function() {
            const slug = this.value.toLowerCase()
                .replace(/[^a-z0-9]+/g, '-')
                .replace(/(^-|-$)/g, '');
            document.getElementById('slug').value = slug;
        });

        // Image preview
        function previewImage(input) {
            const preview = document.getElementById('imagePreview');
            if (input.files && input.files[0]) {
                const reader = new FileReader();
                reader.onload = function(e) {
                    preview.src = e.target.result;
                    preview.style.display = 'block';
                };
                reader.readAsDataURL(input.files[0]);
            } else {
                preview.style.display = 'none';
            }
        }
    </script>
</body>
</html>
