<?php
/**
 * Admin Categories Management
 */
define('ECOMMERCE_ACCESS', true);
require_once '../config/db.php';
require_once '../config/auth.php';

requireAdmin();

$settings = getSettings();
$errors = [];

// Handle delete
if (isset($_GET['delete']) && is_numeric($_GET['delete'])) {
    if (!verifyCSRFToken($_GET['token'] ?? '')) {
        setFlash('error', 'Invalid request');
    } else {
        $categoryId = (int)$_GET['delete'];
        // Set products to uncategorized
        query("UPDATE products SET category_id = NULL WHERE category_id = ?", [$categoryId]);
        query("DELETE FROM categories WHERE id = ?", [$categoryId]);
        setFlash('success', 'Category deleted successfully');
    }
    header('Location: categories.php');
    exit;
}

// Handle add/edit
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    if (!verifyCSRFToken($_POST['csrf_token'] ?? '')) {
        $errors[] = 'Invalid request';
    } else {
        $id = intval($_POST['id'] ?? 0);
        $name = sanitize($_POST['name'] ?? '');
        $slug = sanitize($_POST['slug'] ?? '');
        $description = sanitize($_POST['description'] ?? '');
        $status = isset($_POST['status']) ? 1 : 0;
        
        if (empty($name)) {
            $errors[] = 'Category name is required';
        }
        
        if (empty($slug)) {
            $slug = strtolower(preg_replace('/[^a-zA-Z0-9]+/', '-', $name));
        }
        
        if (empty($errors)) {
            if ($id) {
                // Update
                query(
                    "UPDATE categories SET name = ?, slug = ?, description = ?, status = ? WHERE id = ?",
                    [$name, $slug, $description, $status, $id]
                );
                setFlash('success', 'Category updated successfully');
            } else {
                // Insert
                query(
                    "INSERT INTO categories (name, slug, description, status) VALUES (?, ?, ?, ?)",
                    [$name, $slug, $description, $status]
                );
                setFlash('success', 'Category added successfully');
            }
            header('Location: categories.php');
            exit;
        }
    }
}

// Get categories with product count
$categories = fetchAll("
    SELECT c.*, COUNT(p.id) as product_count 
    FROM categories c 
    LEFT JOIN products p ON c.id = p.category_id 
    GROUP BY c.id 
    ORDER BY c.name
");
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Categories - Admin Panel</title>
    <link rel="stylesheet" href="../assets/style.css">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
    <style>
        :root { --primary: #1e3a5f; --primary-light: #2d4a6f; --success: #10b981; --danger: #ef4444; --gray-100: #f3f4f6; --gray-200: #e5e7eb; --gray-500: #6b7280; --gray-700: #374151; --gray-900: #111827; }
        * { box-sizing: border-box; margin: 0; padding: 0; }
        body { font-family: 'Segoe UI', system-ui, sans-serif; background: var(--gray-100); color: var(--gray-700); }
        .admin-wrapper { display: flex; min-height: 100vh; }
        .sidebar { width: 260px; background: var(--primary); color: #fff; position: fixed; height: 100vh; overflow-y: auto; }
        .sidebar-header { padding: 20px; border-bottom: 1px solid rgba(255,255,255,0.1); }
        .sidebar-header h2 { font-size: 20px; font-weight: 700; }
        .sidebar-header span { font-size: 12px; opacity: 0.7; }
        .sidebar-menu { padding: 20px 0; }
        .menu-item { display: flex; align-items: center; padding: 14px 20px; color: rgba(255,255,255,0.8); text-decoration: none; transition: all 0.3s; gap: 12px; }
        .menu-item:hover, .menu-item.active { background: rgba(255,255,255,0.1); color: #fff; }
        .menu-item i { width: 20px; text-align: center; }
        .main-content { flex: 1; margin-left: 260px; padding: 30px; }
        .page-header { display: flex; justify-content: space-between; align-items: center; margin-bottom: 30px; }
        .page-header h1 { font-size: 28px; color: var(--gray-900); }
        .btn { padding: 10px 20px; border-radius: 8px; text-decoration: none; font-weight: 500; cursor: pointer; border: none; transition: all 0.3s; display: inline-flex; align-items: center; gap: 8px; }
        .btn-primary { background: var(--primary); color: #fff; }
        .btn-primary:hover { background: var(--primary-light); }
        .btn-danger { background: var(--danger); color: #fff; }
        .btn-sm { padding: 6px 12px; font-size: 13px; }
        .card { background: #fff; border-radius: 12px; box-shadow: 0 2px 8px rgba(0,0,0,0.04); margin-bottom: 24px; }
        .card-header { padding: 20px 24px; border-bottom: 1px solid var(--gray-200); }
        .card-header h3 { font-size: 18px; color: var(--gray-900); }
        .card-body { padding: 24px; }
        .table { width: 100%; border-collapse: collapse; }
        .table th, .table td { padding: 14px 16px; text-align: left; border-bottom: 1px solid var(--gray-200); }
        .table th { font-weight: 600; color: var(--gray-500); font-size: 13px; text-transform: uppercase; background: var(--gray-100); }
        .badge { padding: 4px 10px; border-radius: 20px; font-size: 12px; font-weight: 600; }
        .badge-active { background: #d1fae5; color: #059669; }
        .badge-inactive { background: #fee2e2; color: #dc2626; }
        .actions { display: flex; gap: 8px; }
        .form-group { margin-bottom: 16px; }
        .form-group label { display: block; margin-bottom: 6px; font-weight: 500; }
        .form-group input, .form-group textarea { width: 100%; padding: 10px 14px; border: 1px solid var(--gray-200); border-radius: 8px; }
        .form-group input:focus, .form-group textarea:focus { outline: none; border-color: var(--primary); }
        .checkbox-group { display: flex; align-items: center; gap: 8px; }
        .checkbox-group input { width: auto; }
        .grid-2 { display: grid; grid-template-columns: 1fr 2fr; gap: 24px; }
        .alert { padding: 12px 16px; border-radius: 8px; margin-bottom: 20px; }
        .alert-success { background: #d1fae5; color: #059669; }
        .alert-danger { background: #fee2e2; color: #dc2626; }
        @media (max-width: 1024px) { .grid-2 { grid-template-columns: 1fr; } }
        @media (max-width: 768px) {
            .sidebar { width: 100%; height: auto; position: relative; }
            .main-content { margin-left: 0; }
            .admin-wrapper { flex-direction: column; }
        }
    </style>
</head>
<body>
    <div class="admin-wrapper">
        <aside class="sidebar">
            <div class="sidebar-header">
                <h2><?php echo htmlspecialchars($settings['site_name']); ?></h2>
                <span>Admin Panel</span>
            </div>
            <nav class="sidebar-menu">
                <a href="dashboard.php" class="menu-item"><i class="fas fa-tachometer-alt"></i> Dashboard</a>
                <a href="products.php" class="menu-item"><i class="fas fa-box"></i> Products</a>
                <a href="categories.php" class="menu-item active"><i class="fas fa-tags"></i> Categories</a>
                <a href="orders.php" class="menu-item"><i class="fas fa-shopping-cart"></i> Orders</a>
                <a href="users.php" class="menu-item"><i class="fas fa-users"></i> Users</a>
                <a href="settings.php" class="menu-item"><i class="fas fa-cog"></i> Settings</a>
                <a href="logout.php" class="menu-item"><i class="fas fa-sign-out-alt"></i> Logout</a>
            </nav>
        </aside>

        <main class="main-content">
            <div class="page-header">
                <h1>Categories</h1>
            </div>

            <?php echo displayFlash(); ?>

            <div class="grid-2">
                <!-- Add/Edit Form -->
                <div class="card">
                    <div class="card-header">
                        <h3 id="formTitle">Add Category</h3>
                    </div>
                    <div class="card-body">
                        <?php if (!empty($errors)): ?>
                        <div class="alert alert-danger">
                            <?php foreach ($errors as $error): ?>
                            <div><?php echo htmlspecialchars($error); ?></div>
                            <?php endforeach; ?>
                        </div>
                        <?php endif; ?>

                        <form method="POST" id="categoryForm">
                            <?php echo csrfField(); ?>
                            <input type="hidden" name="id" id="categoryId" value="">
                            
                            <div class="form-group">
                                <label for="name">Name *</label>
                                <input type="text" id="name" name="name" required>
                            </div>
                            
                            <div class="form-group">
                                <label for="slug">Slug</label>
                                <input type="text" id="slug" name="slug">
                            </div>
                            
                            <div class="form-group">
                                <label for="description">Description</label>
                                <textarea id="description" name="description" rows="3"></textarea>
                            </div>
                            
                            <div class="form-group">
                                <div class="checkbox-group">
                                    <input type="checkbox" id="status" name="status" value="1" checked>
                                    <label for="status" style="margin: 0;">Active</label>
                                </div>
                            </div>
                            
                            <button type="submit" class="btn btn-primary"><i class="fas fa-save"></i> Save Category</button>
                            <button type="button" class="btn" onclick="resetForm()" style="margin-left: 8px;">Reset</button>
                        </form>
                    </div>
                </div>

                <!-- Categories List -->
                <div class="card">
                    <div class="card-header">
                        <h3>All Categories</h3>
                    </div>
                    <div class="card-body">
                        <table class="table">
                            <thead>
                                <tr>
                                    <th>Name</th>
                                    <th>Products</th>
                                    <th>Status</th>
                                    <th>Actions</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php if (empty($categories)): ?>
                                <tr>
                                    <td colspan="4" style="text-align: center; color: var(--gray-500);">No categories yet</td>
                                </tr>
                                <?php else: ?>
                                <?php foreach ($categories as $cat): ?>
                                <tr>
                                    <td>
                                        <strong><?php echo htmlspecialchars($cat['name']); ?></strong>
                                        <br><small style="color: var(--gray-500);">/<?php echo htmlspecialchars($cat['slug']); ?></small>
                                    </td>
                                    <td><?php echo $cat['product_count']; ?></td>
                                    <td>
                                        <span class="badge badge-<?php echo $cat['status'] ? 'active' : 'inactive'; ?>">
                                            <?php echo $cat['status'] ? 'Active' : 'Inactive'; ?>
                                        </span>
                                    </td>
                                    <td class="actions">
                                        <button class="btn btn-primary btn-sm" onclick="editCategory(<?php echo htmlspecialchars(json_encode($cat)); ?>)">
                                            <i class="fas fa-edit"></i>
                                        </button>
                                        <a href="categories.php?delete=<?php echo $cat['id']; ?>&token=<?php echo generateCSRFToken(); ?>" 
                                           class="btn btn-danger btn-sm"
                                           onclick="return confirm('Delete this category? Products will become uncategorized.')">
                                            <i class="fas fa-trash"></i>
                                        </a>
                                    </td>
                                </tr>
                                <?php endforeach; ?>
                                <?php endif; ?>
                            </tbody>
                        </table>
                    </div>
                </div>
            </div>
        </main>
    </div>

    <script>
        document.getElementById('name').addEventListener('input', function() {
            if (!document.getElementById('categoryId').value) {
                document.getElementById('slug').value = this.value.toLowerCase().replace(/[^a-z0-9]+/g, '-').replace(/(^-|-$)/g, '');
            }
        });

        function editCategory(cat) {
            document.getElementById('formTitle').textContent = 'Edit Category';
            document.getElementById('categoryId').value = cat.id;
            document.getElementById('name').value = cat.name;
            document.getElementById('slug').value = cat.slug;
            document.getElementById('description').value = cat.description || '';
            document.getElementById('status').checked = cat.status == 1;
            document.getElementById('name').focus();
        }

        function resetForm() {
            document.getElementById('formTitle').textContent = 'Add Category';
            document.getElementById('categoryForm').reset();
            document.getElementById('categoryId').value = '';
            document.getElementById('status').checked = true;
        }
    </script>
</body>
</html>
