<?php
/**
 * Authentication Helper Functions
 * E-Commerce Application
 */

// Prevent direct access
if (!defined('ECOMMERCE_ACCESS')) {
    die('Direct access not permitted');
}

// Start session if not started
if (session_status() === PHP_SESSION_NONE) {
    session_start();
}

// Generate CSRF token
function generateCSRFToken() {
    if (empty($_SESSION['csrf_token'])) {
        $_SESSION['csrf_token'] = bin2hex(random_bytes(32));
    }
    return $_SESSION['csrf_token'];
}

// Verify CSRF token
function verifyCSRFToken($token) {
    if (empty($_SESSION['csrf_token']) || empty($token)) {
        return false;
    }
    return hash_equals($_SESSION['csrf_token'], $token);
}

// CSRF token input field
function csrfField() {
    return '<input type="hidden" name="csrf_token" value="' . generateCSRFToken() . '">';
}

// Check if admin is logged in
function isAdminLoggedIn() {
    return isset($_SESSION['admin_id']) && !empty($_SESSION['admin_id']);
}

// Check if user is logged in
function isUserLoggedIn() {
    return isset($_SESSION['user_id']) && !empty($_SESSION['user_id']);
}

// Get current admin
function getCurrentAdmin() {
    if (!isAdminLoggedIn()) {
        return null;
    }
    return fetchOne("SELECT * FROM admins WHERE id = ? AND status = 1", [$_SESSION['admin_id']]);
}

// Get current user
function getCurrentUser() {
    if (!isUserLoggedIn()) {
        return null;
    }
    return fetchOne("SELECT * FROM users WHERE id = ? AND status = 'active'", [$_SESSION['user_id']]);
}

// Admin login
function adminLogin($email, $password) {
    $admin = fetchOne("SELECT * FROM admins WHERE email = ? AND status = 1", [$email]);
    
    if ($admin && password_verify($password, $admin['password'])) {
        $_SESSION['admin_id'] = $admin['id'];
        $_SESSION['admin_name'] = $admin['name'];
        $_SESSION['admin_role'] = $admin['role'];
        
        // Update last login
        query("UPDATE admins SET last_login = NOW() WHERE id = ?", [$admin['id']]);
        
        return true;
    }
    return false;
}

// User login
function userLogin($email, $password) {
    $user = fetchOne("SELECT * FROM users WHERE email = ? AND status = 'active'", [$email]);
    
    if ($user && password_verify($password, $user['password'])) {
        $_SESSION['user_id'] = $user['id'];
        $_SESSION['user_name'] = $user['name'];
        $_SESSION['user_email'] = $user['email'];
        
        // Update last login
        query("UPDATE users SET last_login = NOW() WHERE id = ?", [$user['id']]);
        
        // Transfer session cart to user cart
        transferSessionCart($user['id']);
        
        return true;
    }
    return false;
}

// User registration
function userRegister($name, $email, $password, $phone = null) {
    // Check if email exists
    $existing = fetchOne("SELECT id FROM users WHERE email = ?", [$email]);
    if ($existing) {
        return ['success' => false, 'message' => 'Email already registered'];
    }
    
    // Hash password
    $hashedPassword = password_hash($password, PASSWORD_DEFAULT);
    
    // Insert user
    query(
        "INSERT INTO users (name, email, password, phone) VALUES (?, ?, ?, ?)",
        [$name, $email, $hashedPassword, $phone]
    );
    
    $userId = lastInsertId();
    
    // Auto login
    $_SESSION['user_id'] = $userId;
    $_SESSION['user_name'] = $name;
    $_SESSION['user_email'] = $email;
    
    return ['success' => true, 'user_id' => $userId];
}

// Transfer session cart to user
function transferSessionCart($userId) {
    $sessionId = session_id();
    
    // Get session cart items
    $cartItems = fetchAll("SELECT * FROM cart WHERE session_id = ? AND user_id IS NULL", [$sessionId]);
    
    foreach ($cartItems as $item) {
        // Check if product already in user cart
        $existing = fetchOne(
            "SELECT * FROM cart WHERE user_id = ? AND product_id = ?",
            [$userId, $item['product_id']]
        );
        
        if ($existing) {
            // Update quantity
            query(
                "UPDATE cart SET quantity = quantity + ? WHERE id = ?",
                [$item['quantity'], $existing['id']]
            );
            // Delete session cart item
            query("DELETE FROM cart WHERE id = ?", [$item['id']]);
        } else {
            // Transfer to user
            query(
                "UPDATE cart SET user_id = ?, session_id = NULL WHERE id = ?",
                [$userId, $item['id']]
            );
        }
    }
}

// Admin logout
function adminLogout() {
    unset($_SESSION['admin_id']);
    unset($_SESSION['admin_name']);
    unset($_SESSION['admin_role']);
    session_regenerate_id(true);
}

// User logout
function userLogout() {
    unset($_SESSION['user_id']);
    unset($_SESSION['user_name']);
    unset($_SESSION['user_email']);
    session_regenerate_id(true);
}

// Require admin login
function requireAdmin() {
    if (!isAdminLoggedIn()) {
        header('Location: login.php');
        exit;
    }
}

// Require user login
function requireUser() {
    if (!isUserLoggedIn()) {
        $_SESSION['redirect_after_login'] = $_SERVER['REQUEST_URI'];
        header('Location: login.php');
        exit;
    }
}

// Sanitize input
function sanitize($input) {
    if (is_array($input)) {
        return array_map('sanitize', $input);
    }
    return htmlspecialchars(trim($input), ENT_QUOTES, 'UTF-8');
}

// Validate email
function validateEmail($email) {
    return filter_var($email, FILTER_VALIDATE_EMAIL);
}

// Generate random string
function generateRandomString($length = 16) {
    return bin2hex(random_bytes($length / 2));
}

// Generate order number
function generateOrderNumber() {
    return 'ORD-' . date('Ymd') . '-' . strtoupper(generateRandomString(6));
}

// Flash messages
function setFlash($type, $message) {
    $_SESSION['flash'] = ['type' => $type, 'message' => $message];
}

function getFlash() {
    if (isset($_SESSION['flash'])) {
        $flash = $_SESSION['flash'];
        unset($_SESSION['flash']);
        return $flash;
    }
    return null;
}

function displayFlash() {
    $flash = getFlash();
    if ($flash) {
        $class = $flash['type'] === 'success' ? 'alert-success' : ($flash['type'] === 'error' ? 'alert-danger' : 'alert-info');
        return '<div class="alert ' . $class . '">' . htmlspecialchars($flash['message']) . '</div>';
    }
    return '';
}
